# Hopper-v2 env
# two objectives
# running speed, jumping height

import numpy as np
from gym import utils
from gym.envs.mujoco import mujoco_env
from os import path
import gym
import d4rl
from gym.spaces.box import Box
from moenvs.MujocoEnvs.constraint_env import CMOEnv

class HopperEnv(mujoco_env.MujocoEnv, utils.EzPickle):
    def __init__(self):
        mujoco_env.MujocoEnv.__init__(self, model_path = path.join(path.abspath(path.dirname(__file__)), "assets/hopper.xml"), frame_skip = 5)
        utils.EzPickle.__init__(self)
        self.obj_dim = 2 
        self.action_space_type = "Continuous"
        self.reward_space = np.zeros((self.obj_dim,))
        self.action_space = Box(low=-1.0, high=1.0, shape=(3,), dtype=np.float32)

    def step(self, a):
        a = a * np.array([2, 2, 4])

        posbefore = self.sim.data.qpos[0]
        # if isinstance(a, (np.ndarray)):
        #     a = a[0]
        a = np.clip(a, [-2.0, -2.0, -4.0], [2.0, 2.0, 4.0])
        self.do_simulation(a, self.frame_skip)
        posafter, height, ang = self.sim.data.qpos[0:3]
        alive_bonus = 1.0
        reward_others = alive_bonus - 2e-4 * np.square(a).sum()
        reward_run = 1.5 * (posafter - posbefore) / self.dt + reward_others
        reward_jump = 12. * (height - self.init_qpos[1]) + reward_others
        s = self.state_vector()
        done = not((s[1] > 0.4) and abs(s[2]) < np.deg2rad(90) and abs(s[3]) < np.deg2rad(90) and abs(s[4]) < np.deg2rad(90) and abs(s[5]) < np.deg2rad(90))

        ob = self._get_obs()

        x_velocity = (posafter - posbefore) / self.dt
        return ob, np.array([reward_run, reward_jump],dtype=np.float32), done, {'x_velocity': x_velocity} 
    

    def _get_obs(self):
        return np.concatenate([
            self.sim.data.qpos.flat[1:],
            np.clip(self.sim.data.qvel.flat, -10, 10)
        ])

    def reset_model(self):
        c = 1e-3
        new_qpos = self.init_qpos + self.np_random.uniform(low=-c, high=c, size=self.model.nq)
        new_qpos[1] = self.init_qpos[1]
        new_qvel = self.init_qvel + self.np_random.uniform(low=-c, high=c, size=self.model.nv)
        self.set_state(new_qpos, new_qvel)
        # qpos = self.init_qpos + self.np_random.uniform(
        #     low=-0.005, high=0.005, size=self.model.nq
        # )
        # qvel = self.init_qvel + self.np_random.uniform(
        #     low=-0.005, high=0.005, size=self.model.nv
        # )
        # self.set_state(qpos, qvel)
        return self._get_obs()

    def viewer_setup(self):
        self.viewer.cam.trackbodyid = 2
        self.viewer.cam.distance = self.model.stat.extent * 0.75
        self.viewer.cam.lookat[2] = 1.15
        self.viewer.cam.elevation = -20

class CMOHopperEnv(HopperEnv, CMOEnv):
    def __init__(self):
        CMOEnv.__init__(self, min_episode_cost=0.0, max_episode_cost=250.0, velocity_threshold=0.37315) 
        HopperEnv.__init__(self,)
        self.obj_dim = 3
        self.reward_space = np.zeros((self.obj_dim,))

    def step(self, a):
        ob, rew, _, info = super().step(a)
        x_velocity = info['x_velocity']
        cost = float(x_velocity > self.velocity_threshold)
        rew_vec = np.append(rew, cost)
        rew = rew_vec if self.linear_preference is None else np.dot(rew_vec[:len(self.linear_preference)], self.linear_preference)
        info.update({'cost': cost, 'rew_vec': rew_vec[:-1]})
        done = not self.is_healthy
        return ob, rew, done, info
    
    @property
    def is_healthy(self):
        z, angle = self.sim.data.qpos[1:3]
        state = self.state_vector()[2:]

        min_state, max_state = (-100.0, 100.0)
        min_z, max_z = (0.7, float("inf"))
        min_angle, max_angle = (-0.2, 0.2)

        healthy_state = np.all(np.logical_and(min_state < state, state < max_state))
        healthy_z = min_z < z < max_z
        healthy_angle = min_angle < angle < max_angle

        is_healthy = all((healthy_state, healthy_z, healthy_angle))

        return is_healthy